<?php

/*
 * This class can be used to retrieve messages from an IMAP, POP3, and NNTP server.
 * @author Kiril Kirkov
 * GitHub: https://github.com/kirilkirkov
 * Usage example:
  1. $imap = new Imap();
  2. $connection_result = $imap->connect('{imap.gmail.com:993/imap/ssl}INBOX', 'user@gmail.com', 'secret_password');
     if ($connection_result !== true) {
         echo $connection_result; // Error message!
         exit;
     }
  3. $messages = $imap->getMessages('text'); // Array of messages
 * in $attachments_dir property set directory for attachments
 * in the __destructor set errors log
 */

class Imap {
    private $imapStream;
    private $plaintextMessage;
    private $htmlMessage;
    private $emails;
    private $errors = array();
    private $attachments = array();
    private $attachments_dir = 'attachments';
    private $filename;
    private $encoding;
    private $headers;

    // Connect to the IMAP server
    public function connect($hostname, $username, $password) {
        $this->imapStream = imap_open($hostname, $username, $password);
        
        // Check for connection error
        if (!$this->imapStream) {
            return 'Cannot connect to Mail: ' . imap_last_error();
        }
        return true;
    }

    // Retrieve messages from the inbox
    public function getMessages($type = 'text') {
        $this->attachments_dir = rtrim($this->attachments_dir, '/');
        $stream = $this->imapStream;
        $emails = imap_search($stream, 'ALL');
        $messages = array();

        if ($emails) {
            $this->emails = $emails;
            foreach ($emails as $email_number) {
                $this->attachments = array();
                $uid = imap_uid($stream, $email_number);
                $messages[] = $this->loadMessage($uid, $type);
            }
        } else {
            return array(
                "status" => "error",
                "message" => "No emails found in inbox."
            );
        }

        return array(
            "status" => "success",
            "data" => array_reverse($messages)
        );
    }

    // Get and save attachments
    public function getFiles($r) { 
        $pullPath = $this->attachments_dir . '/' . $r['file'];
        $res = true;
        
        // Check if the file already exists
        if (file_exists($pullPath)) {
            return array("status" => "error", "message" => "File already exists.");
        } elseif (!is_dir($this->attachments_dir)) {
            return array("status" => "error", "message" => 'Directory for attachments not found!');
        } elseif (!is_writable($this->attachments_dir)) {
            return array("status" => "error", "message" => 'Attachments directory is not writable!');
        }

        // Save the attachment if checks pass
        if ($res && !preg_match('/\.php/i', $r['file']) && !preg_match('/\.cgi/i', $r['file']) && !preg_match('/\.exe/i', $r['file'])) {
            if (($filePointer = fopen($pullPath, 'w')) == false) {
                return array("status" => "error", "message" => 'Cannot open file for writing!');
            }

            switch ($r['encoding']) {
                case 3: //base64
                    $streamFilter = stream_filter_append($filePointer, 'convert.base64-decode', STREAM_FILTER_WRITE);
                    break;
                case 4: //quoted-printable
                    $streamFilter = stream_filter_append($filePointer, 'convert.quoted-printable-decode', STREAM_FILTER_WRITE);
                    break;
                default:
                    $streamFilter = null;
            }

            imap_savebody($this->imapStream, $filePointer, $r['uid'], $r['part'], FT_UID);

            if ($streamFilter) {
                stream_filter_remove($streamFilter);
            }

            fclose($filePointer);

            return array("status" => "success", "path" => $pullPath);
        }

        return array("status" => "error", "message" => "Invalid file type.");
    }

    // Load a message by UID
    // In the loadMessage method, replace the decode call with the correct decoding
private function loadMessage($uid, $type) {
    $overview = $this->getOverview($uid);
    $array = array();
    $array['uid'] = $overview->uid;
    // Replace decode() with imap_utf8() for proper decoding
    $array['subject'] = isset($overview->subject) ? imap_utf8($overview->subject) : ''; // Decode subject properly
    $array['date'] = date('Y-m-d h:i:sa', strtotime($overview->date));
    $headers = $this->getHeaders($uid);
    $array['from'] = isset($headers->from) ? $this->processAddressObject($headers->from) : array('');
    $structure = $this->getStructure($uid);
    
    // Process message structure (multipart or plain)
    if (!isset($structure->parts)) {
        $this->processStructure($uid, $structure);
    } else {
        foreach ($structure->parts as $id => $part) {
            $this->processStructure($uid, $part, $id + 1);
        }
    }

    // Set the message content (plain or HTML)
    $array['message'] = $type == 'text' ? $this->plaintextMessage : $this->htmlMessage;
    $array['attachments'] = $this->attachments;
    return $array;
}


    // Process message structure to handle attachments
    private function processStructure($uid, $structure, $partIdentifier = null) {
        $parameters = $this->getParametersFromStructure($structure);
        if ((isset($parameters['name']) || isset($parameters['filename'])) || (isset($structure->subtype) && strtolower($structure->subtype) == 'rfc822')) {
            if (isset($parameters['filename'])) {
                $this->setFileName($parameters['filename']);
            } elseif (isset($parameters['name'])) {
                $this->setFileName($parameters['name']);
            }
            $this->encoding = $structure->encoding;
            $result_save = $this->saveToDirectory($uid, $partIdentifier);
            $this->attachments[] = $result_save;
        } elseif ($structure->type == 0 || $structure->type == 1) {
            $messageBody = isset($partIdentifier) ?
                    imap_fetchbody($this->imapStream, $uid, $partIdentifier, FT_UID | FT_PEEK) : imap_body($this->imapStream, $uid, FT_UID | FT_PEEK);
            $messageBody = $this->decodeMessage($messageBody, $structure->encoding);

            // Convert non-UTF-8 encoded messages
            if (!empty($parameters['charset']) && $parameters['charset'] !== 'UTF-8') {
                if (function_exists('mb_convert_encoding')) {
                    if (!in_array($parameters['charset'], mb_list_encodings())) {
                        if ($structure->encoding === 0) {
                            $parameters['charset'] = 'US-ASCII';
                        } else {
                            $parameters['charset'] = 'UTF-8';
                        }
                    }
                    $messageBody = mb_convert_encoding($messageBody, 'UTF-8', $parameters['charset']);
                } else {
                    $messageBody = iconv($parameters['charset'], 'UTF-8//TRANSLIT', $messageBody);
                }
            }

            // Handle plain or HTML messages
            if (strtolower($structure->subtype) === 'plain' || ($structure->type == 1 && strtolower($structure->subtype) !== 'alternative')) {
                $this->plaintextMessage = '';
                $this->plaintextMessage .= trim(htmlentities($messageBody));
                $this->plaintextMessage = nl2br($this->plaintextMessage);
            } elseif (strtolower($structure->subtype) === 'html') {
                $this->htmlMessage = '';
                $this->htmlMessage .= $messageBody;
            }
        }

        // Process multipart messages
        if (isset($structure->parts)) {
            foreach ($structure->parts as $partIndex => $part) {
                $partId = $partIndex + 1;
                if (isset($partIdentifier))
                    $partId = $partIdentifier . '.' . $partId;
                $this->processStructure($uid, $part, $partId);
            }
        }
    }

    // Save attachments to the specified directory
    private function saveToDirectory($uid, $partIdentifier) { 
        $array = array();
        $array['part'] = $partIdentifier;
        $array['file'] = $this->filename;
        $array['encoding'] = $this->encoding;
        return $array;
    }

    // Decode the message content based on encoding type
    private function decodeMessage($data, $encoding) {
        if (!is_numeric($encoding)) {
            $encoding = strtolower($encoding);
        }
        switch (true) {
            case $encoding === 'quoted-printable':
            case $encoding === 4:
                return quoted_printable_decode($data);
            case $encoding === 'base64':
            case $encoding === 3:
                return base64_decode($data);
            default:
                return $data;
        }
    }

    // Get parameters from the message structure (filename, content-type)
    private function getParametersFromStructure($structure) {
        $parameters = array();
        if (isset($structure->parameters)) {
            foreach ($structure->parameters as $parameter) {
                $parameters[strtolower($parameter->attribute)] = $parameter->value;
            }
        }
        if (isset($structure->dparameters)) {
            foreach ($structure->dparameters as $parameter) {
                $parameters[strtolower($parameter->attribute)] = $parameter->value;
            }
        }
        return $parameters;
    }

    // Set the filename of an attachment
    private function setFileName($filename) {
        $this->filename = $filename;
    }

    // Get the message overview (e.g., subject, date, from)
    private function getOverview($uid) {
        $overview = imap_fetch_overview($this->imapStream, $uid, FT_UID);
        return isset($overview[0]) ? $overview[0] : null;
    }

    // Get headers of a message (from, subject, etc.)
    private function getHeaders($uid) {
        return imap_headerinfo($this->imapStream, $uid);
    }

    // Get the structure of a message (body parts, etc.)
    private function getStructure($uid) {
        return imap_fetchstructure($this->imapStream, $uid, FT_UID);
    }

    // Handle email address objects and parse into array
    private function processAddressObject($address) {
        return array(
            'email' => $address->mailbox . '@' . $address->host,
            'name' => $address->personal
        );
    }

    // Destructor for error logging
    public function __destruct() {
        if (!empty($this->errors)) {
            $errorLog = 'errors.log';
            foreach ($this->errors as $error) {
                // Log errors to a file
                file_put_contents($errorLog, $error . PHP_EOL, FILE_APPEND);
            }
        }
    }
}
